using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// Partial success logon response message requiring client to 
	/// change verify code before continuing.
	/// </summary>
	public sealed class SecurityLogonVerifyCodeChangeRequiredResponseMessage : SecurityBasePartialSuccessResponseMessage
	{
		/// <summary>
		/// VistALink message response type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string ResponseType = "AV.Logon";

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_CHANGE_VERIFY_PART_SCS_ATTRIBUTE_NAME = "changeVerify";
		private const string XMLCONSTS_CHANGE_VERIFY_PART_SCS_ATTRIBUTE_VALUE = "true";
		private const string XMLCONSTS_CVC_HELP_TEXT_PART_SCS_ATTRIBUTE_NAME = "cvcHelpText";

		private string _postSignInText;
		private string _cvcHelpText;

		/// <summary>
		/// Constructor that needs post sign-in text and change verify code help text to create a message.
		/// </summary>
		/// <param name="postSignInText">Server post sign-in text message. Optional - may be null.</param>
		/// <param name="cvcHelpText">Change verify code help text. Expected to be human readable. Optional - may be null.</param>
		public SecurityLogonVerifyCodeChangeRequiredResponseMessage( string postSignInText, string cvcHelpText )
			: this( postSignInText, cvcHelpText, null ) {}

		/// <summary>
		/// Constructor allowing to specify both post sign-in text and informative message. 
		/// </summary>
		/// <param name="postSignInText">Server post sign-in text message. Optional - may be null.</param>
		/// <param name="cvcHelpText">Change verify code help text. Expected to be human readable. Optional - may be null.</param>
		/// <param name="message">Informative message. Optional - may be null.</param>
		public SecurityLogonVerifyCodeChangeRequiredResponseMessage( string postSignInText, string cvcHelpText, string message )
			: base( ResponseType, message )
		{
			_postSignInText = postSignInText;
			_cvcHelpText = cvcHelpText;
		}
		
		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		public SecurityLogonVerifyCodeChangeRequiredResponseMessage( XmlDocument sourceDoc )
			: base( ResponseType, sourceDoc )
		{
			_postSignInText = 
				SecurityLogonNonFailureResponseMessageImplementationMixin.ParsePostSignInTextXml( XmlParseGetResponseElement( sourceDoc ), false );

			XmlElement _partialSuccessElement = XmlParseGetPartialSuccessElement( sourceDoc );

			XmlUtility.ParseCheckRequiredAttributeValue( _partialSuccessElement, XMLCONSTS_CHANGE_VERIFY_PART_SCS_ATTRIBUTE_NAME, XMLCONSTS_CHANGE_VERIFY_PART_SCS_ATTRIBUTE_VALUE );

			if( _partialSuccessElement.HasAttribute( XMLCONSTS_CVC_HELP_TEXT_PART_SCS_ATTRIBUTE_NAME ) )
				_cvcHelpText = _partialSuccessElement.GetAttribute( XMLCONSTS_CVC_HELP_TEXT_PART_SCS_ATTRIBUTE_NAME );
		}

		/// <summary>
		/// Overriden method adding message-specific content to the response element
		/// created by the base class. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlResponseNodeContent( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );			

			SecurityLogonNonFailureResponseMessageImplementationMixin.WritePostSignInTextToXml( writer, _postSignInText, false );

			base.WriteXmlResponseNodeContent( writer );
		}
		
		/// <summary>
		/// Overriden method adding message-specific attributes to the partial success element
		/// created by the base class. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlPartialSuccessAttributes( XmlWriter writer ) 
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteAttributeString( XMLCONSTS_CHANGE_VERIFY_PART_SCS_ATTRIBUTE_NAME, XMLCONSTS_CHANGE_VERIFY_PART_SCS_ATTRIBUTE_VALUE );

			if( _cvcHelpText != null )
				writer.WriteAttributeString( XMLCONSTS_CVC_HELP_TEXT_PART_SCS_ATTRIBUTE_NAME, _cvcHelpText );

			base.WriteXmlPartialSuccessAttributes( writer );
		}

		/// <summary>
		/// Utility XML parsing method utilized to find out whether the supplied VistALink partial success response 
		/// message in XML document partial success type matches this specific message type. It's used in message-to-XML mapping, 
		/// when parsing server XML response and creating an appropriate instance of VistALink message. 
		/// Beware, that source XML document must have 'partialSuccess' element in proper place. Otherwise, XML parsing 
		/// exception will be thrown.  
		/// </summary>
		/// <param name="sourceDoc">Source XML document to analyze.</param>
		/// <returns>True if source XML document </returns>
		public static bool XmlParseDoesPartialSuccessXmlMatchMessage( XmlDocument sourceDoc )
		{
			if( sourceDoc == null )
				throw( new ArgumentNullException( "sourceDoc" ) );

			XmlElement _partialSuccessElement = SecurityBasePartialSuccessResponseMessage.XmlParseGetPartialSuccessElement( sourceDoc );

			return _partialSuccessElement.GetAttribute( XMLCONSTS_CHANGE_VERIFY_PART_SCS_ATTRIBUTE_NAME ) == XMLCONSTS_CHANGE_VERIFY_PART_SCS_ATTRIBUTE_VALUE;
		}


		/// <summary>
		/// Server post sign-in text that should be shown to user. 
		/// </summary>
		public string PostSignInText
		{
			get
			{
				return _postSignInText;
			}
		}

		/// <summary>
		/// Change verify code additional help text. 
		/// </summary>
		public string CvcHelpText
		{
			get
			{
				return _cvcHelpText;
			}
		}
	}
}
